import os
from pathlib import Path
import tkinter as tk
from tkinter import filedialog, messagebox, ttk

class TibetanDuplicateRemover:
    def __init__(self):
        try:
            self.window = tk.Tk()
            self.window.title("藏文重复行清理工具")
            self.window.geometry("800x600")
            self.create_widgets()
        except Exception as e:
            print(f"初始化错误: {str(e)}")
            raise

    def create_widgets(self):
        try:
            # 选择文件夹按钮
            tk.Button(self.window, 
                     text="选择需要处理的文件夹", 
                     command=self.select_folder).pack(pady=10)
            
            # 路径显示标签
            self.path_label = tk.Label(self.window, text="未选择文件夹")
            self.path_label.pack(pady=5)
            
            # 进度条
            self.progress = ttk.Progressbar(self.window, length=300, mode='determinate')
            self.progress.pack(pady=10)
            
            # 处理按钮
            tk.Button(self.window, 
                     text="开始处理", 
                     command=self.process_files).pack(pady=10)
            
            # 日志文本框
            self.log_text = tk.Text(self.window, height=15, width=60)
            self.log_text.pack(pady=10)
            
        except Exception as e:
            print(f"创建界面元素错误: {str(e)}")
            raise

    def select_folder(self):
        try:
            folder_path = filedialog.askdirectory()
            if folder_path:
                self.folder_path = folder_path
                self.path_label.config(text=f"已选择: {folder_path}")
                self.log_message(f"选择文件夹: {folder_path}")
        except Exception as e:
            self.log_message(f"选择文件夹错误: {str(e)}")

    def log_message(self, message):
        try:
            self.log_text.insert(tk.END, message + "\n")
            self.log_text.see(tk.END)
            self.window.update()
        except Exception as e:
            print(f"日志记录错误: {str(e)}")

    def process_file(self, file_path):
        try:
            # 读取文件
            with open(file_path, 'r', encoding='utf-8') as f:
                lines = f.readlines()

            # 处理所有行
            seen_lines = set()
            unique_lines = []
            duplicates = 0

            for line in lines:
                line = line.strip()
                
                # 保留空行
                if not line:
                    unique_lines.append(line)
                    continue
                
                # 对所有非空行进行去重处理
                if line not in seen_lines:
                    seen_lines.add(line)
                    unique_lines.append(line)
                else:
                    duplicates += 1

            # 保存结果
            output_dir = Path(file_path).parent / "cleaned"
            output_dir.mkdir(exist_ok=True)
            
            output_file = output_dir / f"cleaned_{Path(file_path).name}"
            with open(output_file, 'w', encoding='utf-8') as f:
                f.write('\n'.join(unique_lines))

            self.log_message(f"处理文件: {file_path.name}")
            self.log_message(f"删除了 {duplicates} 个重复行")
            
        except Exception as e:
            self.log_message(f"处理文件错误 {file_path}: {str(e)}")

    def process_files(self):
        try:
            if not hasattr(self, 'folder_path'):
                messagebox.showerror("错误", "请先选择文件夹！")
                return

            folder = Path(self.folder_path)
            files = list(folder.glob("*.txt"))
            
            if not files:
                self.log_message("未找到.txt文件！")
                return

            self.log_message(f"开始处理 {len(files)} 个文件...")
            
            for file_path in files:
                self.process_file(file_path)
                
            self.log_message("全部处理完成！")
            messagebox.showinfo("完成", "文件处理完成！")
            
        except Exception as e:
            self.log_message(f"处理文件夹错误: {str(e)}")
            messagebox.showerror("错误", f"处理过程中发生错误: {str(e)}")

    def run(self):
        try:
            self.window.mainloop()
        except Exception as e:
            print(f"运行错误: {str(e)}")

if __name__ == "__main__":
    try:
        app = TibetanDuplicateRemover()
        app.run()
    except Exception as e:
        print(f"程序启动错误: {str(e)}")
